require('dotenv').config();

const express = require('express');
const app = express();
const cors = require('cors');
const fs = require('fs');
const https = require('https');
const cookieParser = require('cookie-parser');
const path = require('path');
const multer = require('multer');

const connectDB = require('./config/db_connection');
const user_module = require('./model/user_module');

const logger = require('./utils/logger');
const logger_payIn_webhook = require('./utils/logger_payIn_webhook');

const {
    startAllArchiveCronJobs,
    startPayinStatusCron,
    startPayoutStatusCron
} = require('./controllers/cronJobController');

const {
    addUserSocket,
    removeUserSocket
} = require('./utils/socketManager');

const PORT = process.env.PORT || 8443;





/* =========================
   SSL CONFIG
========================= */
const sslOptions = {
    key: fs.readFileSync('/etc/letsencrypt/live/playinbyd.com/privkey.pem'),
    cert: fs.readFileSync('/etc/letsencrypt/live/playinbyd.com/fullchain.pem'),
};

const server = https.createServer(sslOptions, app);





/* =========================
   SOCKET.IO
========================= */
const { Server } = require('socket.io');
const io = new Server(server, {
    cors: {
        origin: [
            'https://playinbyd.com',
            'https://admin.playinbyd.com',
            'https://al786.com',
            'http://localhost:3000',
            'http://localhost:5173'
        ],
        credentials: true
    }
});

io.on('connection', (socket) => {
    console.log(`🔌 Socket connected: ${socket.id}`);

    socket.on("userIdentify", async (userId) => {
        console.log("📩 User identified:", userId);

        if (userId) {
            addUserSocket(userId, socket);
            socket.userId = userId;

            const intervalId = setInterval(async () => {
                try {
                    const user = await user_module.findById(userId);
                    if (user) {
                        socket.emit("balanceUpdate", {
                            success: true,
                            balance: user.balance,
                            timestamp: new Date()
                        });
                    }
                } catch (err) {
                    console.error("Balance update error:", err);
                }
            }, 30000);

            socket.userIntervalId = intervalId;
        }
    });

    socket.on('disconnect', () => {
        console.log(`❌ Socket disconnected: ${socket.id}`);

        if (socket.userIntervalId) {
            clearInterval(socket.userIntervalId);
        }

        if (socket.userId) {
            removeUserSocket(socket.userId);
        }
    });
});





/* =========================
   MIDDLEWARES
========================= */
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

app.use(cors({
    origin: [
        'https://playinbyd.com',
         'https://admin.playinbyd.com', 
        'https://al786.com',
        'http://localhost:3000',
        'http://localhost:5173'
    ],
    credentials: true
}));

app.use(cookieParser(process.env.COOKIEPARSER_SECRET));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));





/* =========================
   VIEW ENGINE
========================= */
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));





/* =========================
   REQUEST LOGGER
========================= */
app.use((req, res, next) => {
    console.log('\n||| PATH ||| =>', req.path);
    next();
});





/* =========================
   ROUTES
========================= */
const mobile_route = require('./routes/mobileRoute');
const admin_route = require('./routes/adminRoute');
const payment_route = require('./routes/paymentRoutes');

const { jwt_check_middleware } = require('./middlewares/jwt_check_middleware');

app.use('/', jwt_check_middleware, mobile_route, admin_route, payment_route);





/* =========================
   START SERVICES
========================= */
connectDB();
startPayinStatusCron();
startPayoutStatusCron();
startAllArchiveCronJobs();





/* =========================
   START SERVER
========================= */
server.listen(PORT, () => {
    console.log(`🚀 HTTPS Server running on port ${PORT}`);
});
