// services/dollarbazar.service.js
const axios = require("axios");
const crypto = require("crypto");

const BASE = process.env.DBZ_BASE_URL || "https://collect.dollarbazar.co/collect-service";

// PayIn
const DBZ_PAYIN_API_KEY = process.env.DBZ_API_KEY;
const DBZ_PAYIN_CLIENT_ID = process.env.DBZ_CLIENT_ID;
const DBZ_PAYIN_SALT = process.env.DBZ_SALT;

// PayOut (DIFFERENT CREDENTIALS)
// const DBZ_PAYOUT_API_KEY = process.env.DBZ_PAYOUT_API_KEY;
// const DBZ_PAYOUT_CLIENT_ID = process.env.DBZ_PAYOUT_CLIENT_ID;
// const DBZ_PAYOUT_SALT = process.env.DBZ_PAYOUT_SALT;

if (!DBZ_PAYIN_API_KEY || !DBZ_PAYIN_CLIENT_ID || !DBZ_PAYIN_SALT) {
  console.warn("DollarBazar env vars missing: DBZ_API_KEY, DBZ_CLIENT_ID, DBZ_SALT");
}

function makeHash(amount, orderId) {
  // Concatenate exactly as docs: apiKey=<apiKey>~clientId=<clientId>~amount=<amount>~orderId=<orderId><salt>
  // Some docs included slightly different formatting; below uses exact directive from your doc
  // (concatenate fields in order w/out spaces then append salt)
  const raw = `apiKey=${DBZ_PAYIN_API_KEY}~clientId=${DBZ_PAYIN_CLIENT_ID}~amount=${amount}~orderId=${orderId}${DBZ_PAYIN_SALT}`;
  return crypto.createHash("sha256").update(raw).digest("hex").toUpperCase();
}


async function authenticate(amount, orderId) {
  const hash = makeHash(amount, orderId);
  const body = {
    amount,
    apiKey: DBZ_PAYIN_API_KEY,
    clientId: DBZ_PAYIN_CLIENT_ID,
    orderId,
    hash
  };

  const url = `${BASE}/api/authenticateByHash`;
  const r = await axios.post(url, body, { headers: { "Content-Type": "application/json" }, timeout: 15000 });
  // r.data expected to contain api_token and id_token
  return r.data;
}


async function initiateTransaction(api_token, id_token, payload) {
  // payload must include: amount, orderId, customerEmail, customerMobile, transactionRemark, payerVpa
  const url = `${BASE}/api/upi/initiate/transaction`;
  const headers = {
    "Content-Type": "application/json",
    "api-authorization": `Bearer ${api_token}`,
    "Authorization": `Bearer ${id_token}`
  };
  const r = await axios.post(url, payload, { headers, timeout: 15000 });
  console.log("🚀 ~ initiateTransaction ~ r:", r)
  return r.data;
}

async function checkStatus(api_token, id_token, orderId) {
  const url = `${BASE}/api/upi/check/status/${orderId}`;
  const headers = {
    "api-authorization": `Bearer ${api_token}`,
    "Authorization": `Bearer ${id_token}`
  };
  const r = await axios.get(url, { headers, timeout: 15000 });
  return r.data;
}



module.exports = {
  authenticate,        // Required - gets tokens
  initiateTransaction, // Required - initiates PayIn
  checkStatus          // Required - checks transaction status
};


