// const PaymentGatewayConfig = require('../models/PaymentGatewayConfig');
const dollarbazarService = require('./dollarbazar.service');
const protonpayService = require('./protonpay.service');

class PaymentGatewayManager {
  constructor() {
    this.gateways = {
      'DOLLARBAZAR': dollarbazarService,
      'PROTONPAY': protonpayService
    };
  }

  // Get active gateway for amount
  async getActiveGatewayForAmount(amount) {
    try {
      // Get enabled gateways where amount falls within range
      const gateways = await PaymentGatewayConfig.find({
        enabled: true,
        minAmount: { $lte: amount },
        maxAmount: { $gte: amount }
      }).sort({ priority: 1 });

      if (gateways.length === 0) {
        return null;
      }

      const selectedGateway = gateways[0];
      
      // Update service credentials if needed
      if (selectedGateway.name === 'PROTONPAY') {
        protonpayService.updateCredentials(selectedGateway.credentials);
      }

      return {
        name: selectedGateway.name,
        service: this.gateways[selectedGateway.name],
        config: selectedGateway
      };
    } catch (error) {
      console.error("Error getting active gateway:", error);
      return null;
    }
  }

  // Process payment with selected gateway
  async processPayment(paymentData, gatewayName = null) {
    try {
      const { amount } = paymentData;
      
      let gateway;
      
      if (gatewayName) {
        // Use specified gateway
        gateway = {
          name: gatewayName,
          service: this.gateways[gatewayName]
        };
      } else {
        // Auto-select based on amount
        const activeGateway = await this.getActiveGatewayForAmount(amount);
        if (!activeGateway) {
          throw new Error("No active payment gateway available for this amount");
        }
        gateway = activeGateway;
      }

      console.log(`Processing payment with ${gateway.name} gateway`);

      // Process based on gateway type
      if (gateway.name === 'DOLLARBAZAR') {
        return await this.processDollarBazarPayment(paymentData);
      } else if (gateway.name === 'PROTONPAY') {
        return await this.processProtonPayPayment(paymentData);
      } else {
        throw new Error(`Unsupported gateway: ${gateway.name}`);
      }

    } catch (error) {
      console.error("Payment processing error:", error);
      throw error;
    }
  }

  // Process DollarBazar payment
  async processDollarBazarPayment(paymentData) {
    const { amount, orderId, customerEmail, customerMobile, transactionRemark, payerVpa } = paymentData;

    // Step 1: Authenticate
    const authResp = await dollarbazarService.authenticate(amount, orderId);
    if (!authResp || !authResp.api_token || !authResp.id_token) {
      throw new Error("DollarBazar authentication failed");
    }

    const { api_token, id_token } = authResp;

    // Step 2: Initiate transaction
    const txPayload = {
      amount: String(amount),
      orderId,
      customerEmail,
      customerMobile,
      transactionRemark,
      payerVpa: payerVpa || ""
    };

    const initResp = await dollarbazarService.initiateTransaction(api_token, id_token, txPayload);

    return {
      gateway: 'DOLLARBAZAR',
      success: initResp.success,
      orderId,
      intentUrl: initResp.data?.intentUrl,
      transactionId: initResp.data?.transactionId,
      gatewayResponse: initResp
    };
  }

  // Process ProtonPay payment
  async processProtonPayPayment(paymentData) {
    const { 
      amount, 
      orderId, 
      userId, 
      userName, 
      phoneNumber, 
      email,
      returnUrl 
    } = paymentData;

    // Step 1: Initialize payment
    const initData = {
      amount,
      orderId,
      userId,
      userName,
      phoneNumber,
      email,
      returnUrl
    };

    const initResp = await protonpayService.initializePayment(initData);

    if (!initResp.success) {
      throw new Error(`ProtonPay initialization failed: ${initResp.error}`);
    }

    const { payinId } = initResp;

    // Step 2: Get UPI link
    const upiResp = await protonpayService.getUpiLink(orderId, payinId);

    if (!upiResp.success) {
      throw new Error(`ProtonPay UPI link failed: ${upiResp.error}`);
    }

    return {
      gateway: 'PROTONPAY',
      success: true,
      orderId,
      payinId,
      upiLink: upiResp.upiLink,
      gatewayResponse: {
        initialize: initResp.gatewayResponse,
        upiLink: upiResp.gatewayResponse
      }
    };
  }

  // Refresh transaction status
  async refreshTransactionStatus(transaction) {
    const { payment_method, tx_request_number } = transaction;
    
    if (payment_method === 'DOLLARBAZAR') {
      const authResp = await dollarbazarService.authenticate(transaction.amount, tx_request_number);
      if (!authResp || !authResp.api_token || !authResp.id_token) {
        throw new Error("Authentication failed with payment gateway");
      }

      const statusResp = await dollarbazarService.checkStatus(authResp.api_token, authResp.id_token, tx_request_number);
      return this.mapDollarBazarStatus(statusResp);
      
    } else if (payment_method === 'PROTONPAY') {
      const statusResp = await protonpayService.checkPaymentStatus(tx_request_number);
      return this.mapProtonPayStatus(statusResp);
    } else {
      throw new Error(`Unsupported payment method: ${payment_method}`);
    }
  }

  // Map DollarBazar status to your fund_status
  mapDollarBazarStatus(statusResp) {
    if (!statusResp.success) {
      return { newStatus: 0, gatewayData: statusResp };
    }

    const data = statusResp.data;
    if (data.orderStatus === "PAID" && data.bankReference) {
      return { 
        newStatus: 1, // Approved
        gatewayData: statusResp,
        bankReference: data.bankReference
      };
    } else if (data.status === "FAIL" || data.orderStatus === "FAIL") {
      return { newStatus: 2, gatewayData: statusResp }; // Failed
    } else {
      return { newStatus: 0, gatewayData: statusResp }; // Pending
    }
  }

  // Map ProtonPay status to your fund_status
  mapProtonPayStatus(statusResp) {
    if (!statusResp.success) {
      return { newStatus: 0, gatewayData: statusResp };
    }

    const status = statusResp.status;
    if (status === 'success' || status === 'SUCCESS') {
      return { 
        newStatus: 1, // Approved
        gatewayData: statusResp
      };
    } else if (status === 'failed' || status === 'FAILED' || status === 'rejected') {
      return { newStatus: 2, gatewayData: statusResp }; // Failed
    } else {
      return { newStatus: 0, gatewayData: statusResp }; // Pending
    }
  }
}

module.exports = new PaymentGatewayManager();