const mongoose = require('mongoose');
const wallet_trans_history_module = require('../model/wallet_trans_history_module');
const wallet_trans_archive_history_module = require('../model/wallet_trans_archive_history_module');
const fix_values_module = require('../model/fix_values_module'); // Import your fix values model

let isWalletArchiveRunning = false;
let lastWalletRunTime = null;
const MIN_INTERVAL_BETWEEN_RUNS = 30000; // 30 seconds minimum between runs

/**
 * Get archive days from fix_values configuration
 */
const getArchiveDays = async () => {
    try {
        const fixValues = await fix_values_module.findOne({ app_id: 1 });
        return fixValues?.archive_history_days || 30; // Default to 30 days if not set
    } catch (error) {
        console.error('❌ Error getting archive days:', error);
        return 30; // Default fallback
    }
};

/**
 * Move old wallet transactions to archive collection based on configured days
 */
const archiveOldWalletTransactions = async () => {
    // Prevent multiple concurrent runs
    if (isWalletArchiveRunning) {
        console.log('⏳ Wallet archive job is already running, skipping...');
        return { success: false, message: 'Wallet archive job is already running' };
    }

    // Rate limiting - don't run too frequently
    const now = Date.now();
    if (lastWalletRunTime && (now - lastWalletRunTime) < MIN_INTERVAL_BETWEEN_RUNS) {
        console.log('⏳ Wallet archive job ran too recently, skipping...');
        return { success: false, message: 'Wallet archive job ran too recently' };
    }

    isWalletArchiveRunning = true;
    lastWalletRunTime = now;
    const session = await mongoose.startSession();

    try {
        console.log('🔄 Starting wallet transaction archive process...');

        // Get archive days from configuration
        const archiveDays = await getArchiveDays();

        // Calculate date based on configured days
        const cutoffDate = new Date();
        cutoffDate.setDate(cutoffDate.getDate() - archiveDays);
        const cutoffISO = cutoffDate.toISOString();

        console.log(`📅 Archiving wallet transactions older than: ${cutoffISO} (${archiveDays} days)`);

        // Find wallet transactions older than configured days
        const oldWalletTransactions = await wallet_trans_history_module.find({
            insert_date: { $lte: cutoffISO }
        }).session(session);

        console.log(`📊 Found ${oldWalletTransactions.length} wallet transactions to archive`);

        if (oldWalletTransactions.length === 0) {
            console.log('✅ No old wallet transactions found to archive');
            return {
                success: true,
                message: 'No wallet transactions to archive',
                archivedCount: 0,
                archiveDays: archiveDays
            };
        }

        let result;
        await session.withTransaction(async () => {
            // Convert to plain objects and prepare for archive
            const archiveData = oldWalletTransactions.map(transaction => {
                const transactionObj = transaction.toObject();
                delete transactionObj._id; // Let MongoDB create new _id for archive
                return transactionObj;
            });

            // Insert into archive collection
            await wallet_trans_archive_history_module.insertMany(archiveData, { session });

            // Delete from original collection
            await wallet_trans_history_module.deleteMany({
                _id: { $in: oldWalletTransactions.map(transaction => transaction._id) }
            }).session(session);

            result = {
                success: true,
                message: `Successfully archived ${oldWalletTransactions.length} wallet transactions`,
                archivedCount: oldWalletTransactions.length,
                archiveDays: archiveDays
            };

            console.log(`✅ Wallet archive completed: ${oldWalletTransactions.length} transactions moved to archive collection (${archiveDays} days threshold)`);
        });

        return result;

    } catch (error) {
        console.error('❌ Error in wallet transaction archive process:', error);
        return {
            success: false,
            message: 'Wallet archive process failed',
            error: error.message
        };
    } finally {
        await session.endSession();
        isWalletArchiveRunning = false;
    }
};

module.exports = {
    archiveOldWalletTransactions
};










// const mongoose = require('mongoose');
// const wallet_trans_history_module = require('../model/wallet_trans_history_module');
// const wallet_trans_archive_history_module = require('../model/wallet_trans_archive_history_module');

// let isWalletArchiveRunning = false;
// let lastWalletRunTime = null;
// const MIN_INTERVAL_BETWEEN_RUNS = 30000; // 30 seconds minimum between runs

// /**
//  * Move 1-month-old wallet transactions to archive collection
//  */
// const archiveOldWalletTransactions = async () => {
//     // Prevent multiple concurrent runs
//     if (isWalletArchiveRunning) {
//         console.log('⏳ Wallet archive job is already running, skipping...');
//         return { success: false, message: 'Wallet archive job is already running' };
//     }

//     // Rate limiting - don't run too frequently
//     const now = Date.now();
//     if (lastWalletRunTime && (now - lastWalletRunTime) < MIN_INTERVAL_BETWEEN_RUNS) {
//         console.log('⏳ Wallet archive job ran too recently, skipping...');
//         return { success: false, message: 'Wallet archive job ran too recently' };
//     }

//     isWalletArchiveRunning = true;
//     lastWalletRunTime = now;
//     const session = await mongoose.startSession();

//     try {
//         console.log('🔄 Starting wallet transaction archive process...');

//         // Calculate date 1 month ago
//         const todayDate = new Date().toISOString();
//         const oneMonthAgo = new Date();
//         oneMonthAgo.setMonth(oneMonthAgo.getMonth() - 1);
//         const oneMonthAgoISO = oneMonthAgo.toISOString();

//         console.log(`~ archiveOldWalletTransactions ~ todayDate: ${todayDate}`);
//         console.log(`📅 Archiving wallet transactions older than: ${oneMonthAgoISO}`);

//         // Find wallet transactions older than 1 month
//         const oldWalletTransactions = await wallet_trans_history_module.find({
//             insert_date: { $lte: oneMonthAgo }
//         }).session(session);

//         console.log(`📊 Found ${oldWalletTransactions.length} wallet transactions to archive`);

//         if (oldWalletTransactions.length === 0) {
//             console.log('✅ No old wallet transactions found to archive');
//             return {
//                 success: true,
//                 message: 'No wallet transactions to archive',
//                 archivedCount: 0
//             };
//         }

//         let result;
//         await session.withTransaction(async () => {
//             // Convert to plain objects and prepare for archive
//             const archiveData = oldWalletTransactions.map(transaction => {
//                 const transactionObj = transaction.toObject();
//                 delete transactionObj._id; // Let MongoDB create new _id for archive
//                 return transactionObj;
//             });

//             // Insert into archive collection
//             await wallet_trans_archive_history_module.insertMany(archiveData, { session });

//             // Delete from original collection
//             await wallet_trans_history_module.deleteMany({
//                 _id: { $in: oldWalletTransactions.map(transaction => transaction._id) }
//             }).session(session);

//             result = {
//                 success: true,
//                 message: `Successfully archived ${oldWalletTransactions.length} wallet transactions`,
//                 archivedCount: oldWalletTransactions.length
//             };

//             console.log(`✅ Wallet archive completed: ${oldWalletTransactions.length} transactions moved to archive collection`);
//         });

//         return result;

//     } catch (error) {
//         console.error('❌ Error in wallet transaction archive process:', error);
//         return {
//             success: false,
//             message: 'Wallet archive process failed',
//             error: error.message
//         };
//     } finally {
//         await session.endSession();
//         isWalletArchiveRunning = false;
//     }
// };




// module.exports = {
//     archiveOldWalletTransactions
// };